<?php
/**
 * User Has Tag
 *
 * @package     AutomatorWP\Integrations\FluentCRM\Filters\User_Has_Tag
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_FluentCRM_User_Has_Tag_Filter extends AutomatorWP_Integration_Filter {

    public $integration = 'fluentcrm';
    public $filter = 'fluentcrm_user_has_tag';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_filter( $this->filter, array(
            'integration'       => $this->integration,
            'label'             => __( 'User has tag', 'automatorwp-pro' ),
            'select_option'     => __( 'User has <strong>tag</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Condition. %2$s: Tag. */
            'edit_label'        => sprintf( __( '%1$s %2$s', 'automatorwp-pro' ), '{condition}', '{tag}' ),
            /* translators: %1$s: Condition. %2$s: Tag. */
            'log_label'         => sprintf( __( '%1$s %2$s', 'automatorwp-pro' ), '{condition}', '{tag}'),
            'options'           => array(
                'tag' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'tag',
                    'option_default'    => __( 'Select a tag', 'automatorwp-pro' ),
                    'name'              => __( 'Tag:', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Tag ID', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_fluentcrm_get_tags',
                    'options_cb'        => 'automatorwp_fluentcrm_options_cb_tag',
                    'default'           => ''
                ) ),
                'condition' => array(
                    'from' => 'condition',
                    'fields' => array(
                        'condition' => array(
                            'name' => __( 'Condition:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'equal'             => __( 'equal to', 'automatorwp-pro' ),
                                'not_equal'         => __( 'not equal to', 'automatorwp-pro' ),
                            ),
                            'default' => 'equal'
                        )
                    )
                ),
            ),
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_filter    True if user deserves filter, false otherwise
     * @param stdClass  $filter             The filter object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $filter_options     The filter's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_filter( $deserves_filter, $filter, $user_id, $event, $filter_options, $automation ) {

        global $wpdb;
        
        // Shorthand
        $tag_id = absint( $filter_options['tag'] );
        $condition = $filter_options['condition'];

        // Bail if empty tag to assign
        if( empty( $tag_id ) ) {
            return;
        }

        $tag = $wpdb->get_row( $wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}fc_tags WHERE id = %s",
            $tag_id
        ) );

        // Bail if tag not exists
        if( ! $tag ) {
            $this->result = __( 'Filter not passed. Tag does not exist.', 'automatorwp-pro' );
            return false;
        }

        $subscriber = automatorwp_fluentcrm_get_subscriber( $user_id );

        // Bail if subscriber not exists
        if( ! $subscriber ) {
            $this->result = __( 'Filter not passed. User is not a subscriber.', 'automatorwp-pro' );
            return false;
        }


        switch( $condition ) {
            case 'equal':
                // Don't deserve if user does not have the tag
                if( ! $subscriber->hasAnyTagId( array( $tag_id ) ) ) {
                    $this->result = sprintf( __( 'Filter not passed. User does not meet the condition tag %1$s "%2$s".', 'automatorwp-pro' ),
                        automatorwp_utilities_get_condition_label( $condition ),
                        $tag->title
                    );
                    return false;
                }
                break;
            case 'not_equal':
                // Don't deserve if user has the tag
                if( $subscriber->hasAnyTagId( array( $tag_id ) ) ) {
                    $this->result = sprintf( __( 'Filter not passed. User does not meet the condition tag %1$s "%2$s".', 'automatorwp-pro' ),
                    automatorwp_utilities_get_condition_label( $condition ),
                    $tag->title
                    );
                    return false;
                }
                break;
        }

        $this->result = sprintf( __( 'Filter passed. User has tag %1$s "%2$s".', 'automatorwp-pro' ),
            automatorwp_utilities_get_condition_label( $condition ),
            $tag->title
        );

        return $deserves_filter;

    }

}

new AutomatorWP_FluentCRM_User_Has_Tag_Filter();